/*:
 * @target MZ
 * @plugindesc 装備品に設定したスキルを指定ターンごとに自動発動（ターン終了時）させるプラグイン（複数同時発動対応） v1.2
 * @author ChatGPT
 *
 * @help
 * ■装備品（武器・防具）のメモ欄に以下のタグを記述します：
 *
 *   <AutoSkill: スキルID, 開始ターン, 間隔ターン>
 *
 * 例：
 *   <AutoSkill: 15, 1, 3>
 *     → スキルID15を 1,4,7,...ターン目の「ターン終了時」に自動発動
 *
 * ■仕様
 * - 対象は「戦闘参加メンバーのアクター」と「敵キャラ」
 * - 非戦闘メンバー（控え）は対象外
 * - 戦闘不能（HP0）のバトラーは発動しない
 * - 発動タイミングは「ターン終了時」
 * - ターゲットはスキルの範囲に従う
 *   - 単体対象スキルの場合はランダムターゲット
 * - 複数装備で同じターンに複数発動条件を満たしても、キューに積んで順番に全て発動
 * - 消費MP/TPは通常通り支払われます（ただし強制行動の性質上、MP不足でも発動自体は止まりません）
 *
 * ※プラグインコマンドやパラメータはありません。
 */

(() => {
  "use strict";

  // ============================================================
  // メモタグ解析（武器・防具・敵にも一応対応）
  // ============================================================

  const autoSkillTag = /<AutoSkill:\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*>/i;

  const _DataManager_extractMetadata = DataManager.extractMetadata;
  DataManager.extractMetadata = function (item) {
    _DataManager_extractMetadata.call(this, item);

    if (!item || !item.note) return;

    const autoSkills = [];
    const lines = item.note.split(/[\r\n]+/);
    for (const line of lines) {
      const match = autoSkillTag.exec(line);
      if (match) {
        const skillId = Number(match[1]);
        const startTurn = Number(match[2]);
        const interval = Number(match[3]);

        // interval <= 0 は無視（毎ターンは interval=1 を使ってね）
        if (skillId > 0 && interval > 0) {
          autoSkills.push({ skillId, startTurn, interval });
        }
      }
    }
    if (autoSkills.length > 0) {
      item._autoSkills = autoSkills;
    }
  };

  // ============================================================
  // バトラーにターンカウンタを持たせる
  // ============================================================

  const _Game_Battler_onBattleStart = Game_Battler.prototype.onBattleStart;
  Game_Battler.prototype.onBattleStart = function () {
    _Game_Battler_onBattleStart.call(this);
    this._autoSkillTurnCount = 0;
  };

  // ============================================================
  // BattleManager: オートスキル用キュー
  // ============================================================

  BattleManager._autoSkillQueue = [];

  // 戦闘開始時にキュー初期化
  const _BattleManager_startBattle = BattleManager.startBattle;
  BattleManager.startBattle = function () {
    this._autoSkillQueue = [];
    _BattleManager_startBattle.call(this);
  };

  // 逃走/敗北/勝利等で戦闘終了した後も念のためクリア
  const _BattleManager_endBattle = BattleManager.endBattle;
  BattleManager.endBattle = function (result) {
    this._autoSkillQueue = [];
    _BattleManager_endBattle.call(this, result);
  };

  // 毎フレーム更新でキューを処理（強制行動が空いたら次を実行）
  const _BattleManager_update = BattleManager.update;
  BattleManager.update = function () {
    _BattleManager_update.call(this);
    this.processAutoSkillQueue();
  };

  BattleManager.processAutoSkillQueue = function () {
    if (!this._autoSkillQueue || this._autoSkillQueue.length === 0) return;

    // 戦闘処理が忙しい時は待つ
    if (this.isBusy()) return;

    // すでに強制行動が走ってるなら待つ
    if (this._actionForcedBattler) return;

    // 次のキューを取り出す
    const next = this._autoSkillQueue.shift();
    if (!next) return;

    const { user, skillId, targetIndex } = next;

    // 行動者が無効ならスキップ
    if (!user || !user.isAlive()) return;

    // スキルが存在しないならスキップ
    const skill = $dataSkills[skillId];
    if (!skill) return;

    // 強制行動登録（ここで初めて forceAction する）
    user.forceAction(skillId, targetIndex);
    this.forceAction(user);
  };

  // ============================================================
  // ターン終了時：カウントアップ＆判定→キューに積む
  // ============================================================

  const _BattleManager_endTurn = BattleManager.endTurn;
  BattleManager.endTurn = function () {
    // 通常のターン終了処理を先に実行
    _BattleManager_endTurn.call(this);

    // ターンカウント更新
    this.updateAutoSkillTurns();

    // 発動判定してキューに積む
    this.enqueueAutoSkills();
  };

  BattleManager.updateAutoSkillTurns = function () {
    $gameParty.battleMembers().forEach((b) => {
      if (b) b._autoSkillTurnCount = (b._autoSkillTurnCount || 0) + 1;
    });
    $gameTroop.members().forEach((b) => {
      if (b) b._autoSkillTurnCount = (b._autoSkillTurnCount || 0) + 1;
    });
  };

  BattleManager.enqueueAutoSkills = function () {
    const battlers = this.allBattleMembers();

    for (const battler of battlers) {
      if (!battler || !battler.isAlive()) continue;

      const turn = battler._autoSkillTurnCount || 0;
      if (turn <= 0) continue;

      const configs = this.collectAutoSkills(battler);
      if (configs.length === 0) continue;

      for (const cfg of configs) {
        const { skillId, startTurn, interval } = cfg;
        if (interval <= 0) continue;

        if (turn >= startTurn && ((turn - startTurn) % interval === 0)) {
          this.enqueueAutoSkillAction(battler, skillId);
        }
      }
    }
  };

  BattleManager.collectAutoSkills = function (battler) {
    const list = [];

    if (battler.isActor()) {
      battler.equips().forEach((equip) => {
        if (equip && equip._autoSkills) list.push(...equip._autoSkills);
      });
    } else if (battler.isEnemy()) {
      // 敵にもメモタグを使えるようにしておく（不要なら消してOK）
      const enemyData = battler.enemy();
      if (enemyData && enemyData._autoSkills) list.push(...enemyData._autoSkills);
    }

    return list;
  };

  // ============================================================
  // キューに積む（ターゲットはここで決定して固定）
  // ============================================================

  BattleManager.enqueueAutoSkillAction = function (user, skillId) {
    const skill = $dataSkills[skillId];
    if (!skill) return;

    const targets = this.makeAutoSkillTargets(user, skill);
    if (!targets || targets.length === 0) return;

    // 単体なら targets[0] がランダム1体。全体でもここでは index 用に先頭を使う。
    // ※全体スキルでも targetIndex は「-1（全体）」が理想だが、MZのforceActionはindexが必要。
    //   ここでは「対象側ユニット先頭」を与え、実際の適用はスキル範囲に従って処理される想定。
    const target = targets[0];
    const targetIndex = this.autoTargetIndex(user, target);

    this._autoSkillQueue.push({ user, skillId, targetIndex });
  };

  // ============================================================
  // ターゲット決定
  // ============================================================

  BattleManager.makeAutoSkillTargets = function (user, skill) {
    const action = new Game_Action(user);
    action.setSkill(skill.id);
    const scope = skill.scope;

    // 単体相当（敵単体/敵ランダム/味方単体/味方ランダム）
    const isSingle =
      scope === 1 || // 敵単体
      scope === 2 || // 敵単体（ランダム）
      scope === 7 || // 味方単体
      scope === 8;   // 味方単体（ランダム）

    if (isSingle) {
      const candidates = action.isForOpponent()
        ? user.opponentsUnit().aliveMembers()
        : user.friendsUnit().aliveMembers();

      if (candidates.length === 0) return [];
      return [candidates[Math.randomInt(candidates.length)]];
    }

    // 全体・自分などは候補を返す（キュー登録の便宜上）
    if (action.isForOpponent()) {
      return user.opponentsUnit().aliveMembers();
    } else if (action.isForUser()) {
      return [user];
    } else {
      return user.friendsUnit().aliveMembers();
    }
  };

  // ============================================================
  // ターゲットIndex決定
  // ============================================================

  BattleManager.autoTargetIndex = function (user, target) {
    if (!target) return -1;

    if (user.isActor()) {
      if (target.isEnemy()) return target.index();
      return $gameParty.battleMembers().indexOf(target);
    } else {
      // 敵側
      if (target.isActor()) return target.index();
      return target.index();
    }
  };

})();
